package gov.va.genisis2.ms.controller;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.owasp.esapi.ESAPI;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.config.EnableJpaRepositories;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import gov.va.genisis2.ms.data.model.ConceptMapping;
import gov.va.genisis2.ms.service.IConceptMappingService;
import gov.va.genisis2.ts.common.dto.ConceptMappingDTO;
import gov.va.genisis2.ts.common.enums.ErrorEnum;
import gov.va.genisis2.ts.common.exception.TSDuplicateDataException;
import gov.va.genisis2.ts.common.exception.TSInernalSystemException;
import gov.va.genisis2.ts.common.exception.TSNoDataFoundException;

/**
 * Spring REST Controller class for Terminology Mapping service
 *
 * @author PII
 *
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/")
@EnableJpaRepositories("gov.va.genisis2.ms.data.repository")
public class MappingServiceController {

	private static final Logger LOGGER = LogManager.getLogger(MappingServiceController.class);

	@Autowired
	private IConceptMappingService conceptMappingService;

	@RequestMapping(value = "/mappings", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	@ResponseBody
	public ResponseEntity<ConceptMappingDTO> getConceptMappingByConceptUri(@RequestParam(value = "conceptUri", required = true) String conceptUri) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getConceptMappingByConceptUri");
		}

		ConceptMappingDTO conceptMappingDto = null;
		try {
			conceptMappingDto = conceptMappingService.getConceptMappingByConceptUri(conceptUri);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		if (null == conceptMappingDto) {
			LOGGER.error(String.format(ErrorEnum.CONCEPT_MAPPING_NO_DATA_FOUND_ERROR.getErrorMessage(), conceptUri));
			throw new TSNoDataFoundException(String.format(ErrorEnum.CONCEPT_MAPPING_NO_DATA_FOUND_ERROR.getErrorMessage(), conceptUri));
		} else {
			return new ResponseEntity<ConceptMappingDTO>(conceptMappingDto, HttpStatus.OK);
		}
	}

	@SuppressWarnings("rawtypes")
	@RequestMapping(value = "/mappings", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	@ResponseBody
	public ResponseEntity createConceptMapping(@RequestBody ConceptMappingDTO conceptMappingDto) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("createConceptMapping");
		}

		try {
			ConceptMapping existingConceptMapping = conceptMappingService.removeConceptMapping(conceptMappingDto);
			conceptMappingService.createConceptMapping(conceptMappingDto, existingConceptMapping);
		} catch (TSDuplicateDataException ex) {
			throw ex;
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), e);
			throw new TSInernalSystemException(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage());
		}

		return new ResponseEntity<>(HttpStatus.CREATED);
	}

	/**
	 * No data found exception.
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSNoDataFoundException.class })
	public ResponseEntity<Object> exceptionHandler(TSNoDataFoundException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML( ex.getMessage( )), HttpStatus.NOT_FOUND);
	}

	/**
	 * Duplicate data exception for create API calls
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSDuplicateDataException.class })
	public ResponseEntity<Object> exceptionHandler(TSDuplicateDataException ex) {
		return new ResponseEntity<Object>(
				ESAPI.encoder().encodeForHTML(ex.getMessage()), HttpStatus.BAD_REQUEST);
	}

	/**
	 * Internal System Error exception
	 *
	 * @param ex
	 *            the ex
	 * @return the response entity
	 */
	@ExceptionHandler({ TSInernalSystemException.class })
	public ResponseEntity<Object> exceptionHandler(TSInernalSystemException ex) {
		return new ResponseEntity<Object>(ErrorEnum.INTERNAL_SYS_ERROR.getErrorMessage(), HttpStatus.INTERNAL_SERVER_ERROR);
	}
}
